//*************************************************************************************************
//
//	Description:
//		dirt2D.fx - simple 2D shader ( based on basic2d.fx )
//		takes 2D position, UV coordinates, and vertex colour as inputs
//              uses parameters to scale and offset UVs - but in the pixel shader, to avoid clipping issues
//              uses another vec4 parameter to generate noise UVs
//              uses final parameter to control jitter
// 
//
//	<P> Copyright (c) 2008 Blimey! Games Ltd. All rights reserved.
//
//	Author: 
//		Robert Dibley

//
// Textures
//

#include "stddefs.fxh"

texture EffectTexture : TEXTURE < bool appEdit = true; >;
texture NoiseTexture : TEXTURE < bool appEdit = true; >;

float4	UV_scaleoffset < bool appEdit = true; >;
float4	noise_control < bool appEdit = true; >;
float4	jitter < bool appEdit = true; >;	// only actually need first two

//-----------------------------------------------------------------------
//
// Samplers
//

sampler2D EffectMap : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="EffectTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "None";
	string AddressU  = "Clamp";
	string AddressV  = "Clamp";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < EffectTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = Linear;
	MagFilter = Linear;
	MipFilter = None;
	AddressU = Clamp;
	AddressV = Clamp;
#endif
};


sampler2D NoiseMap : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="NoiseTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "None";
	string AddressU  = "Clamp";
	string AddressV  = "Clamp";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < NoiseTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = Linear;
	MagFilter = Linear;
	MipFilter = None;
	AddressU = Clamp;
	AddressV = Clamp;
#endif
};


//-----------------------------------------------------------------------
//
// Vertex Shader(s)
//

// Input structure
struct VSINPUT
{
	float2 position : POSITION;
	float2 texcoord : TEXCOORD0;
	float4 colour : COLOR0;
};


// Output structure (also input to the fragment shader)
struct VSOUTPUT
{
	float4 position		: POSITION;			// View-coords position
	float2 texcoord		: TEXCOORD0;			// UV coords for texture channel 0
	float4 colour 		: COLOR0;
};


//-----------------------------------------------------------------------
//
// Vertex shader code
//

VSOUTPUT DirtVertexShader( VSINPUT _input )
{
	VSOUTPUT _output = ( VSOUTPUT )0;

	_output.position.xy = _input.position.xy;
	_output.position.z = 0.0f;
	_output.position.w = 1.0f;
	
	_output.texcoord = _input.texcoord;
	_output.colour = _input.colour;

	return _output;
}
			


//-----------------------------------------------------------------------
//
// Fragment Shader(s)
//

// Output structure
struct PSOUTPUT
{
	COLOUR_OUTPUT_TYPE Colour : COLOR0;
};



//-----------------------------------------------------------------------
//
// Fragment shader code
//

PSOUTPUT DirtFragmentShader( VSOUTPUT _input )
{
	PSOUTPUT _output = ( PSOUTPUT )0;

	// Read textures - starting with noise
	float2  noiseUV = lerp( noise_control.xy, noise_control.zw, saturate(_input.texcoord.x) );
	float4  noise = tex2D( NoiseMap, noiseUV);

	// now use one channel of noise to vary vertical position
	float offset = jitter.x * (noise.x - 0.5f);

	float2	uv = saturate(float2(_input.texcoord.x, _input.texcoord.y + offset));

	float4	dirtColour = tex2D( EffectMap, uv );

	float colScale = 1.0f - jitter.y * (noise.y - 0.5f);	// allows variable strength of scratch along length

	dirtColour *= _input.colour;		// allows vertex interpolated colour to vary strength
	dirtColour *= colScale;

	_output.Colour.xyz = dirtColour.xyz;
	_output.Colour.w = 1.0f;	// note sure what we will be using the alpha for, if anything

	return _output;
}


//-----------------------------------------------------------------------
//
// Technique(s)
//


// standard behaviour - add the effect, no multiplies or anything
technique EffectAdd
<
	string normalBehaviour	= "ERMB_RENDER";
	string normalTechnique	= "EffectAdd";
	int    normalDeferredID	= 0;
>
{
	pass Pass0
	{
		ZEnable = false;
		ZWriteEnable = false;
		AlphaBlendEnable = true;
#ifdef _PS3_
		CullFaceEnable=false;
		BlendFunc=int2(One, One);
		BlendEquation=int(FuncAdd);
		VertexShader = compile sce_vp_rsx DirtVertexShader();
		PixelShader = compile sce_fp_rsx DirtFragmentShader();
#else		
		CullMode = None;
		DestBlend = ONE;
		SrcBlend = ONE;
		BlendOp = ADD;
		VertexShader = compile vs_3_0 DirtVertexShader();
		PixelShader = compile ps_3_0 DirtFragmentShader();
#endif
	}
}

// wiping behaviour - for cleaning e.g. windscreen wipers
technique EffectSub
<
	string normalBehaviour	= "ERMB_RENDER";
	string normalTechnique	= "EffectSub";
	int    normalDeferredID	= 0;
>
{
	pass Pass0
	{
		ZEnable = false;
		ZWriteEnable = false;
		AlphaBlendEnable = true;

#if defined (_PS3_)
		CullFaceEnable=false;
		BlendFunc=int2(One, One);
		BlendEquation=int(FuncSubtract);
		VertexShader = compile sce_vp_rsx DirtVertexShader();
		PixelShader = compile sce_fp_rsx DirtFragmentShader();
#else		
		CullMode = None;
		SrcBlend = ONE;
		DestBlend = ONE;
		BlendOp = REVSUBTRACT;	
		VertexShader = compile vs_3_0 DirtVertexShader();
		PixelShader = compile ps_3_0 DirtFragmentShader();
#endif

	}
}

